clear all
close all

%----------
% define two arbitrary covariant base vectors
%
% gcov1 and gcov2
%----------

ancov1 = 0.034*pi;    % arbitrary
ancov2 = 0.334*pi;    % arbitrary

lcov1 = 1.4;  % arbitrary
lcov2 = 1.8;  % arbitrary

gcov1(1) = lcov1*cos(ancov1); gcov1(2) = lcov1*sin(ancov1);
gcov2(1) = lcov2*cos(ancov2); gcov2(2) = lcov2*sin(ancov2);

%---
% covariant components of the metric tensor
%---

covmet(1,1) = gcov1(1)*gcov1(1) + gcov1(2)*gcov1(2);
covmet(1,2) = gcov1(1)*gcov2(1) + gcov1(2)*gcov2(2);
covmet(2,1) = covmet(1,2);
covmet(2,2) = gcov2(1)*gcov2(1) + gcov2(2)*gcov2(2);

covg = det(covmet);

%---
% Compute gcontra1 and gcontra2
%---

ancon1 = ancov2 - 0.5*pi;
ancon2 = ancov1 + 0.5*pi;

lcon1 = 2.4; % arbitrary
lcon2 = 1.2; % arbitrary

gcon1(1) = lcon1*cos(ancon1); gcon1(2) = lcon1*sin(ancon1);
gcon2(1) = lcon2*cos(ancon2); gcon2(2) = lcon2*sin(ancon2);

%---
% normalize the contravariant base vectors
% so that cov1*con1 = 1,  cov2*con2 = 1,
%---

norm1 = gcov1(1)*gcon1(1) + gcov1(2)*gcon1(2);
norm2 = gcov2(1)*gcon2(1) + gcov2(2)*gcon2(2);

gcon1(1) = gcon1(1)/norm1;
gcon1(2) = gcon1(2)/norm1;

gcon2(1) = gcon2(1)/norm2;
gcon2(2) = gcon2(2)/norm2;

%---
% another way of computing the contravariant base vectors
% explicitly in terms of gcov1 and gcov2
%---

gcon1A = ( covmet(2,2)*gcov1 - covmet(1,2)*gcov2)/covg
gcon2A = (-covmet(1,2)*gcov1 + covmet(1,1)*gcov2)/covg

%---
% another way of computing the cvariant base vectors
% explicitly in terms of gcon1 and gcon2
%---

gcov1A = covmet(1,1)*gcon1 + covmet(1,2)*gcon2;
gcov2A = covmet(1,2)*gcon1 + covmet(2,2)*gcon2;

[gcov1  gcov2;
 gcov1A gcov2A;
 gcon1  gcon2;
 gcon1A gcon2A]

%---
% contravariant components of the metric tensor
%---

conmet(1,1) = gcon1(1)*gcon1(1) + gcon1(2)*gcon1(2);
conmet(1,2) = gcon1(1)*gcon2(1) + gcon1(2)*gcon2(2);
conmet(2,1) = conmet(1,2);
conmet(2,2) = gcon2(1)*gcon2(1) + gcon2(2)*gcon2(2);

[inv(covmet) conmet]

%---
% the metric tensor is the identity matrix
%---

for m=1:2
 for n=1:2
  gmet1(m,n) =  ...
   conmet(1,1) * gcov1(m)*gcov1(n) ...
  +conmet(1,2) * gcov1(m)*gcov2(n) ...
  +conmet(2,1) * gcov2(m)*gcov1(n) ...
  +conmet(2,2) * gcov2(m)*gcov2(n);

  gmet2(m,n) =  ...
   covmet(1,1) * gcon1(m)*gcon1(n) ...
  +covmet(1,2) * gcon1(m)*gcon2(n) ...
  +covmet(2,1) * gcon2(m)*gcon1(n) ...
  +covmet(2,2) * gcon2(m)*gcon2(n);

  gmet3(m,n) =  ...
   gcon1(m)*gcov1(n) ...
  +gcon2(m)*gcov2(n);

 end
end

[gmet1 gmet2 gmet3]
