function [mats,mat,rhs] = pois_fds_DDDD  ...
  ...
   (ax,bx ...
   ,ay,by ...
   ,Nx,Ny ...
   ,gxx,gxy,gyy ...
   ,vx,vy ...
   ,source ...
   ,w,q,z,v ...
  )

%==============================================
% Generate a finite-difference linear system 
% for the Poisson equation
% inside a rectangle confined in ax<x<bx, ay<y<by
%
% Equation is:    Lalp(f) + source = 0
%
% System is :     mat * x = rhs
%
% Boundary conditions:   f = w   at x = ax
%                        f = q   at x = bx
%                        f = z   at y = ay
%                        f = v   at y = by
%
% The system is generated by the method of impulses
% described by Pozrikidis (2008)
%
% SYMBOLS:
% -------
%
% Nx...	  intervals in x direction
% Ny...   intervals in y direction
% mats... system size
% mat..	  finite-difference matrix
% rhs..   right-hand side
%
% unknown vector is comprised of sequential values (i,j):
% (horizonal and then up)
%
%  2,2   3,2   4,2    ...  Nx-1,2   Nx,2 
%  2,3   3,3   4,3    ...  Nx-1,3   Nx,2
%  ...
%
%  2,Ny  3,Ny  4,Ny   ...  Nx-1,Ny  Nx,Ny
%
%=====================================

%-------------
% preparations
%-------------

 Dx = (bx-ax)/Nx;
 Dy = (by-ay)/Ny;

%------------------
% more preparations
%------------------

 Dx2  = 2.0D0*Dx;
 Dy2  = 2.0D0*Dy;

 Dxs  = Dx^2;
 Dys  = Dy^2;

 vp = Dx/Dy;
 vps = vp*vp;

%-------------------
% initialize to zero
%-------------------

 for j=2:Ny
   for i=2:Nx
     f(i,j) = 0.0;
   end
 end

%------------------------------
% Dirichlet boundary conditions
%------------------------------

  for j=2:Ny
    f(   1,j) = w(j);     % left side
    f(Nx+1,j) = q(j);     % right side
  end

  for i=2:Nx
    f(i,   1) = z(i);     % down
    f(i,Ny+1) = v(i);     % up
  end

%---
% right-hand side
%---

 p = 0;     % counter

 for j=2:Ny
   for i=2:Nx
     p = p+1;
     R =       gxx(i,j)*(f(i-1,j)-2.0*f(i,j)+f(i+1,j)) ...
         + vps*gyy(i,j)*(f(i,j-1)-2.0*f(i,j)+f(i,j+1)) ...
         + 0.5*vp*gxy(i,j)*(f(i+1,j+1)-f(i-1,j+1) ...
                           -f(i+1,j-1)+f(i-1,j-1)) ...
         + 0.5*Dx*   vx(i,j)*(f(i+1,j)-f(i-1,j)) ...
         + 0.5*Dx*vp*vy(i,j)*(f(i,j+1)-f(i,j-1)) ...
         + Dxs*source(i,j);
      rhs(p) = - R;
   end
 end

 mats = p;         % system size

%-------------------------------
% scan row-by-row to compute mat
%-------------------------------

      t = 0;    % counter

      for s=2:Ny
        for l=2:Nx

          f(l,s) = 1.0D0;  %  impulse
          t = t+1;

          p = 0;      % counter

          for j=2:Ny
            for i=2:Nx
             p = p+1;
         R =       gxx(i,j)*(f(i-1,j)-2.0*f(i,j)+f(i+1,j)) ...
         + vps*gyy(i,j)*(f(i,j-1)-2.0*f(i,j)+f(i,j+1)) ...
         + 0.5*vp*gxy(i,j)*(f(i+1,j+1)-f(i-1,j+1) ...
                           -f(i+1,j-1)+f(i-1,j-1)) ...
         + 0.5*Dx*   vx(i,j)*(f(i+1,j)-f(i-1,j)) ...
         + 0.5*Dx*vp*vy(i,j)*(f(i,j+1)-f(i,j-1)) ...
         + Dxs*source(i,j);

             mat(p,t) = R+rhs(p);

            end
          end

        f(l,s) = 0.0;   % reset

        end
      end

%-----
% done
%-----

 return
