clear all
close all

%============================
% Solve the Poisson equation
% in a domain generated by
% conformal mapping of a
% rectangle a1 < xi < b1
%          a2 < eta < b2
%============================

%---
% parameters
%---

N1 = 2*16;
N2 = 2*16;

a1 = -0.5;
b1 =  0.5;

a2 = -0.5;
b2 =  0.5;

%---
% prepare
%---

Dx1 = (b1-a1)/N1;
Dx2 = (b2-a2)/N2;

for i=1:N1+1
  xdiv1(i) = a1+(i-1.0)*Dx1;
end

for j=1:N2+1
  xdiv2(j) = a2+(j-1.0)*Dx2;
end

%---
% grid in the xi-eta plane
%---

for j=1:N2+1
 for i=1:N1+1
  x1(i,j) = xdiv1(i);
  x2(i,j) = xdiv2(j);
 end
end

%---
% prepare
%---

Dx1 = (b1-a1)/N1;
Dx2 = (b2-a2)/N2;

for i=1:N1+1
  xdiv1(i) = a1+(i-1.0)*Dx1;
end

%-------------
% plot the grid in the xi-eta plane
%-------------

iplot1=0;
iplot1=1;

if(iplot1==1)

figure(1)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-0.6 0.6 -0.6 0.6])

for j=1:N2+1
  plot(x1(1:N1+1,j),x2(1:N1+1,j),'k')
end

for i=1:N1+1
  plot(x1(i,1:N2+1),x2(i,1:N2+1),'k')
end

end

%---
% grid in the xy plane
%---

im = sqrt(-1);
srt = sqrt(2);

for j=1:N2+1
 for i=1:N1+1
  xi = x1(i,j);
  et = x2(i,j);

  %---
  % conformal mapping
  %---

  zet = xi + im*et;
  mapp = (1.0+1.0*im-0.9*zet^2)*(1.0-0.8*zet^4)*zet/srt;
  x(i,j) = real(mapp);
  y(i,j) = imag(mapp);

 end
end

%---
% plot the grid in the xy plane
%---

iplot2=0;
iplot2=1;

if(iplot2==1)

figure(2)
hold on
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-1.1 1.1 -1.1 1.1])

for j=1:N2+1
  plot(x(1:N1+1,j),y(1:N1+1,j),'k')
end

for i=1:N1+1
  plot(x(i,1:N2+1),y(i,1:N2+1),'k')
end

end

%---
% covariant base vectors
% by central differences
%---

for j=2:N2
 for i=2:N1

   gcov1_x(i,j) = (x(i+1,j)-x(i-1,j))/(2.0*Dx1);
   gcov1_y(i,j) = (y(i+1,j)-y(i-1,j))/(2.0*Dx1);

   gcov2_x(i,j) = (x(i,j+1)-x(i,j-1))/(2.0*Dx2);
   gcov2_y(i,j) = (y(i,j+1)-y(i,j-1))/(2.0*Dx2);

 end
end

% perimeter values by interpolation

gcov1_x = perimeter(N1,N2,gcov1_x);
gcov1_y = perimeter(N1,N2,gcov1_y);
gcov2_x = perimeter(N1,N2,gcov2_x);
gcov2_y = perimeter(N1,N2,gcov2_y);

%---
% plot the covariant base vectors
%---

iplot31=0;
iplot31=1;

if(iplot31==1)

figure(31)
hold on
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-1.1 1.1 -1.1 1.1])

fc = 0.75*Dx1;

for j=1:N2+1
 for i=1:N1+1
  arrx = fc*gcov1_x(i,j);
  arry = fc*gcov1_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  plot(gaidaros(:,1), gaidaros(:,2),'k')
  arrx = fc*gcov2_x(i,j);
  arry = fc*gcov2_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  plot(gaidaros(:,1), gaidaros(:,2),'k')
 end
end

end

%---
% covariant metric coefficients
%---

for j=1:N2+1
 for i=1:N1+1
  covmet11(i,j) = gcov1_x(i,j)*gcov1_x(i,j) ...
                 +gcov1_y(i,j)*gcov1_y(i,j);
  covmet12(i,j) = gcov1_x(i,j)*gcov2_x(i,j) ...
                 +gcov1_y(i,j)*gcov2_y(i,j);
  covmet21(i,j) = covmet12(i,j);
  covmet22(i,j) = gcov2_x(i,j)*gcov2_x(i,j) ...
                 +gcov2_y(i,j)*gcov2_y(i,j);
  covg(i,j) = covmet11(i,j)*covmet22(i,j)-covmet12(i,j)^2;
  srcovg(i,j) = sqrt(covg(i,j));
 end
end

%---
% contravariant base vectors
%---

for j=1:N2+1
 for i=1:N1+1
  gcon1_x(i,j) = ( covmet22(i,j)*gcov1_x(i,j) ...
                  -covmet12(i,j)*gcov2_x(i,j))/covg(i,j);
  gcon1_y(i,j) = ( covmet22(i,j)*gcov1_y(i,j) ...
                  -covmet12(i,j)*gcov2_y(i,j))/covg(i,j);
  gcon2_x(i,j) = (-covmet12(i,j)*gcov1_x(i,j) ...
                  +covmet11(i,j)*gcov2_x(i,j))/covg(i,j);
  gcon2_y(i,j) = (-covmet12(i,j)*gcov1_y(i,j) ...
                  +covmet11(i,j)*gcov2_y(i,j))/covg(i,j);
 end
end

%---
% plot the contravariant base vectors
%---

iplot32=0;
iplot32=1;

if(iplot32==1)

figure(32)
hold on
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-1.1 1.1 -1.1 1.1])

fc = 0.25*Dx1;

for j=1:N2+1
 for i=1:N1+1
  arrx = fc*gcon1_x(i,j);
  arry = fc*gcon1_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  plot(gaidaros(:,1), gaidaros(:,2),'k')
  arrx = fc*gcon2_x(i,j);
  arry = fc*gcon2_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  plot(gaidaros(:,1), gaidaros(:,2),'k')
 end
end

end

%---
% contravariant metric coefficients
%---

Ido = 0;
Ido = 1;

if(Ido==1)

for j=1:N2+1
 for i=1:N1+1
  conmet11(i,j) = gcon1_x(i,j)*gcon1_x(i,j) ...
                + gcon1_y(i,j)*gcon1_y(i,j);
  conmet12(i,j) = gcon1_x(i,j)*gcon2_x(i,j) ...
                + gcon1_y(i,j)*gcon2_y(i,j);
  conmet21(i,j) = conmet12(i,j);
  conmet22(i,j) = gcon2_x(i,j)*gcon2_x(i,j) ...
                + gcon2_y(i,j)*gcon2_y(i,j);
  cong(i,j) = conmet11(i,j)*conmet22(i,j)-conmet12(i,j)^2;
 end
end

end

%---
% plot g
%---

iplot35=0;
iplot35=1;

if(iplot35==1)

figure(35)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
view(-12,30)

%surf(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),g(1:N1+1,1:N2+1))
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),covg(1:N1+1,1:N2+1))

end

%---
% plot gcon_11, gcon_12, gcon_22
%---

iplot11=0;
iplot11=1;

if(iplot11==1)

figure(11)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^{\xi\xi}','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet11(1:N1+1,1:N2+1))
view(-12,30)

end

iplot12=0;
iplot12=1;

if(iplot12==1)

figure(12)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^(\xi\eta)','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet12(1:N1+1,1:N2+1))
view(-12,30)

end

iplot22=0;
iplot22=1;

if(iplot22==1)

figure(22)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^{\eta\eta}','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet22(1:N1+1,1:N2+1))
view(-12,30)

end

%-------------
% compute the effective velocities v1 and v2
%-------------

for j=2:N2
 for i=2:N1
  v1(i,j) = (conmet11(i+1,j)*srcovg(i+1,j) ...
            -conmet11(i-1,j)*srcovg(i-1,j))/(2*Dx1) ...
           +(conmet12(i,j+1)*srcovg(i,j+1) ...
            -conmet12(i,j-1)*srcovg(i,j-1))/(2*Dx2);
  v2(i,j) = (conmet21(i+1,j)*srcovg(i+1,j) ...
            -conmet21(i-1,j)*srcovg(i-1,j))/(2*Dx1) ...
           +(conmet22(i,j+1)*srcovg(i,j+1) ...
            -conmet22(i,j-1)*srcovg(i,j-1))/(2*Dx2);
  v1(i,j) = v1(i,j)/srcovg(i,j);
  v2(i,j) = v2(i,j)/srcovg(i,j);
 end
end

v1 = perimeter(N1,N2,v1);
v2 = perimeter(N1,N2,v2);

%---
% plot v1 and v2
%---

iplot81 = 0;
iplot81 = 1;

if(iplot81==1)

figure(81)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('v^\xi','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),v1(1:N1+1,1:N2+1))
view(-64,42)

figure(82)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('v^\eta','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),v2(1:N1+1,1:N2+1))
view(-64,42)

end

%=================
% POISSON EQUATION
%=================

%--------------------
% boundary conditions
%--------------------

  for j=1:N2+1
    w(j) = 0.0;  % example
    q(j) = 0.0;  % example
  end

  for i=1:N1+1
    z(i) = 0.0;  % example
    v(i) = 0.0;  % example
  end

%---
% source term
%---

  for i=1:N1+1
    for j=1:N2+1
      source(i,j) = 5.0; % example
    end
  end

%---

[mats,mat,rhs] = pois_fds_DDDD  ...
  ...
   (a1,b1 ...
   ,a2,b2 ...
   ,N1,N2 ...
   ,conmet11,conmet12,conmet22 ...
   ,v1,v2 ...
   ,source ...
   ,w,q,z,v ...
   );

%-----------------
% printing session
%-----------------

%  mat


  sol = rhs/mat';

%---
% distribute the solution
%---

 p = 0;     % counter

  for j=2:N2
   for i=2:N1
     p = p+1;
     f(i,j) = sol(p);
   end
  end

  for j=1:N2+1
    f(   1,j) = w(j);
    f(N1+1,j) = q(j);
  end

  for i=1:N1+1
    f(i,   1) = z(i);
    f(i,N2+1) = v(i);
  end

%---
% plot
%---

figure(91)
mesh(x,y,f)
axis equal
axis([-1 1.1 -1.1 1.1 0 0.5])
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
zlabel('f','fontsize',15)
set(gca,'fontsize',15)
box on
view(-32,32)
