clear all
close all

%---
% parameters
%---

a = 1;  % x semi-axis of innermost ellipse
b = 2;  % y semi-axis of innermost ellipse

%---
% compute xi_0 and A
%---

xi0 = atanh(a/b);
snhxi0 = sinh(xi0);
cshxi0 = cosh(xi0);
A = a/snhxi0;

ximax = log(4.0*b/A);

Np = 2;
Nxi = Np*8;
Net = Np*16;

%---
% grid
%---

 Dxi = (ximax-xi0)/Nxi;

 for i=1:Nxi+1
  xi(i) = xi0+(i-1.0)*Dxi;
  snhxi(i) = sinh(xi(i));
  cshxi(i) = cosh(xi(i));
 end

 Det = 2.0*pi/Net;

 for j=1:Net+1
  eta(j) = (j-1.0)*Det;
  cseta(j) = cos(eta(j));
  sneta(j) = sin(eta(j));
 end

%---
% grid points
%---

 for i=1:Nxi+1
  for j=1:Net+1
   x(i,j) = A*snhxi(i)*cseta(j);
   y(i,j) = A*cshxi(i)*sneta(j);
   J(i,j) = A*sqrt(cshxi(i)^2-sneta(j)^2);
  end
 end

%---
% polar angle around the orgin (theta)
%---

 xcnt = 0.0;
 ycnt = 0.0;

 for j=1:Net+1
  rr = sqrt((x(1,j)-xcnt)^2+(y(1,j)-ycnt)^2);
  theta(j) = acos((x(1,j)-xcnt)/rr);
  if(y(1,j)<0.0)
   theta(j) = 2*pi-theta(j);
  end
 end

 theta(Net+1) = 2*pi+theta(1);

%---
% plot
%---

 figure(1)
 hold on;

 x = real(x);
 y = real(y);
 J = real(J);

 for j=1:Net+1
  plot(x(:,j),y(:,j),'k')
 end

 for i=1:Nxi+1
  plot(x(i,:),y(i,:),'r')
 end

% plot(x(1,1),y(1,1),'bo')

 xlabel('x','fontsize',15);
 ylabel('y','fontsize',15);
 set(gca,'fontsize',15)
 axis square
 axis([-5 5 -5 5])
 box on

%----
% Poisson equation
%----

 for j=1:Net+1
  for i=1:Nxi+1
   s(i,j) = 1.0;  % source
  end
 end

 for j=1:Net+1
  for i=1:Nxi+1
   f(i,j) = 0.0;  % initialize
  end
 end

 for j=1:Net+1
  f(1,    j) = 0.0;  % inner boundary condition
  f(Nxi+1,j) = 0.3;  % outer boundary condition
 end

 beta = (Dxi/Det)^2;

%---
% iterations
%---

 tolerance = 0.0001;

 for iter = 1:200

  for i=2:Nxi  % periodicity condition
    f(i,1)     = f(i,Net+1);
    f(i,Net+2) = f(i,2);
  end

 corrmax = 0.0;
 fc1 = 1/(2.0*(1+beta));
 Dxis = Dxi^2;

 for j=2:Net+1
  for i=2:Nxi

   fold = f(i,j);

   f(i,j) = fc1*( f(i+1,j)+f(i-1,j) ...
          + beta*(f(i,j+1)+f(i,j-1)) ...
          + Dxis*J(i,j)^2*s(i,j) );

   corr = abs(fold-f(i,j));
   if(corr>corrmax) corrmax=corr; end
  end
 end

%corrmax
 if(corrmax<tolerance) break; end

 end

 iter

%===
% plot
%===

 figure(2)
 hold on;

 f = real(f);

%---
% plot
%---

%for j=1:Net+1
% plot3(x(:,j),y(:,j),f(1:Nxi+1,j),'k')
%end

%for i=1:Nxi+1
% plot3(x(i,:),y(i,:),f(i,1:Net+1),'r')
%end

%plot(x(1,1),y(1,1),'bo')

%surf(x,y,f(1:Nxi+1,1:Net+1))
mesh(x,y,f(1:Nxi+1,1:Net+1))

 xlabel('x','fontsize',15);
 ylabel('y','fontsize',15);
 zlabel('f','fontsize',15);
 set(gca,'fontsize',15)
 axis square
 axis([-5 5 -5 5 0 1.2])
 box on
 view(-14,20)

