function [mats,mat,rhs] = pois_fds_PPDD  ...
  ...
   (ax,bx ...
   ,ay,by ...
   ,Nx,Ny ...
   ,gxx,gxy,gyy ...
   ,vx,vy ...
   ,source ...
   ,z,v ...
   )

%==============================================
% Generate a finite-difference linear system 
% for the Poisson equation
% inside a rectangle confined in
% ax<x<bx, ay<y<by
%
% Equation is:    Lalp(f) + source = 0
%
% System is :     mat * x = rhs
%
% Boundary conditions:   f = periodic   at x = ax
%                        f = periodic   at x = bx
%                        f = z          at y = ay
%                        f = v          at y = by
%
% The system is generated by the method of impulses
%
% SYMBOLS:
% -------
%
% Nx...	  intervals in x direction
% Ny...   intervals in y direction
% mats... system size
% mat..	  finite-difference matrix
% rhs..   right-hand side
%
% unknown vector is comprised of sequential values (i,j):
% (horizonal and then up)
%
%  2,2   3,2   4,2    ...  Nx,2   Nx+1,2 
%  2,3   3,3   4,3    ...  Nx,3   Nx+1,2
%  ...
%
%  2,Ny  3,Ny  4,Ny   ...  Nx,Ny  Nx+1,Ny
%=====================================

%-------------
% preparations
%-------------

 Dx = (bx-ax)/Nx;
 Dy = (by-ay)/Ny;

%------------------
% more preparations
%------------------

 Dx2 = 2.0*Dx;
 Dy2 = 2.0*Dy;

 Dxs = Dx^2;
 Dys = Dy^2;

 vp  = Dx/Dy;
 vps = vp*vp;

%-------------------
% initialize to zero
%-------------------

 for j=1:Ny+1
   for i=1:Nx+2
     f(i,j) = 0.0;
   end
 end

%------------------------------
% Dirichlet boundary conditions
% at bottom and top
%------------------------------

  for i=1:Nx+2
    f(i,   1) = z(i);     % bottom
    f(i,Ny+1) = v(i);     % top
  end

%---
% wrap
%---

  for j=1:Ny+1
    f(1,j) = f(Nx+1,j);
    f(Nx+2,j) = f(2,j);
  end

%---
% right-hand side
%---

 p = 0;     % counter

 for j=2:Ny
   for i=2:Nx+1
     p = p+1;
     R =       gxx(i,j)*(f(i-1,j)-2.0*f(i,j)+f(i+1,j)) ...
         + vps*gyy(i,j)*(f(i,j-1)-2.0*f(i,j)+f(i,j+1)) ...
         + 0.5*vp*gxy(i,j)*(f(i+1,j+1)-f(i-1,j+1) ...
                           -f(i+1,j-1)+f(i-1,j-1)) ...
         + 0.5*Dx*   vx(i,j)*(f(i+1,j)-f(i-1,j)) ...
         + 0.5*Dx*vp*vy(i,j)*(f(i,j+1)-f(i,j-1)) ...
         + Dxs*source(i,j);
      rhs(p) = - R;
   end
 end

 mats = p;         % system size

%-------------------------------
% scan row-by-row to compute mat
%-------------------------------

      t = 0;    % counter

      for s=2:Ny
        for l=2:Nx+1

          f(l,s) = 1.0;  %  impulse

          for j=2:Ny   % wrap
           f(1,j) = f(Nx+1,j);
           f(Nx+2,j) = f(2,j);
          end

          t = t+1;

          p = 0;      % counter

          for j=2:Ny
            for i=2:Nx+1
             p = p+1;
             R =     gxx(i,j)*(f(i-1,j)-2.0*f(i,j)+f(i+1,j)) ...
               + vps*gyy(i,j)*(f(i,j-1)-2.0*f(i,j)+f(i,j+1)) ...
               + 0.5*vp*gxy(i,j)*(f(i+1,j+1)-f(i-1,j+1) ...
                                 -f(i+1,j-1)+f(i-1,j-1)) ...
               + 0.5*Dx*   vx(i,j)*(f(i+1,j)-f(i-1,j)) ...
               + 0.5*Dx*vp*vy(i,j)*(f(i,j+1)-f(i,j-1)) ...
               + Dxs*source(i,j);
               mat(p,t) = R+rhs(p);
            end
          end

          f(l,s) = 0.0;   % reset

          for j=2:Ny  % wrap
            f(1,j) = f(Nx+1,j);
            f(Nx+2,j) = f(2,j);
          end

        end
      end

%-----
% done
%-----

 return
