clear all
close all

%============================
% Solution of the Poisson equation
% with Dirichlet boundary conditions
% in curvilinear coordinates
% in a channel-like domain
%
% xi = x^1, eta =x^2
%
% xi varies in [ax, bx]
% eta varies in [ay, by]
%============================

%---
% channel geometrical parameters
%---

ichannel = 2; % annulus
ichannel = 1; % channel

L = 1.3;
h = 0.45;

%---
% amplitudes
%---

alow = 0.4;
ahig = 0.4;

alow = 0.0;
ahig = 0.0;

ahig = 0.2;
alow = 0.1;

phasehig = 0.25*pi;
phasehig = 0.0*pi;

%---
% parameters
%---

N1 = 1*16;
N2 = 1*16;

if(ichannel==2)
 N1 = 2*16;
 N2 = 1*16;
end

a1 = -0.5;
b1 =  0.5;

a2 = -0.5;
b2 =  0.5;

%---
% prepare
%---

Dx1 = (b1-a1)/N1;
Dx2 = (b2-a2)/N2;

for i=1:N1+2
  xdiv1(i) = a1+(i-1.0)*Dx1;
end

for j=1:N2+2
  xdiv2(j) = a2+(j-1.0)*Dx2;
end

for j=1:N2+2
 for i=1:N1+2
  x1(i,j) = xdiv1(i);
  x2(i,j) = xdiv2(j);
 end
end

%-------------
% plot the grid in the x1-x2 plane
%-------------

iplot1=1;
iplot1=0;

if(iplot1==1)

figure(1)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-0.6 0.6 -0.6 0.6])

for j=1:N2+1
  plot(x1(1:N1+1,j),x2(1:N1+1,j),'k')
end

for i=1:N1+1
  plot(x1(i,1:N2+1),x2(i,1:N2+1),'k')
end

end

%---
% grid in the xy plane
%---

cf1 = 0.0;
cf2 = 0.0;

for j=1:N2+2
 for i=1:N1+2

  x(i,j) = L*x1(i,j);
  wlow(i) = h + alow*(cos(2*pi*x1(i,j)) ...
              +   cf1*cos(4*pi*x1(i,j)) ...
              +   cf2*cos(6*pi*x1(i,j)) );
  whig(i) = h+ahig*cos(2*pi*x1(i,j)+phasehig);
  y(i,j) = (x2(i,j)-0.5)*wlow(i) + (x2(i,j)+0.5)*whig(i);

  if(ichannel==2) % annulus

   im = sqrt(-1);
   r = exp(1.0+y(i,j)/h);
   theta = 2*pi*x(i,j)/L;
   x(i,j) = r*cos(theta);
   y(i,j) = r*sin(theta);

  end

 end
end

%---
% plot grid in the xy plane
%---

iplot2=0;
iplot2=1;

if(iplot2==1)

figure(2)
hold on
xlabel('x/L','fontsize',15)
ylabel('y/L','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
%axis([-0.5 0.5 -(h+alow)/L (h+ahig)/L])
% axis([-0.5 0.5 -1.6*(h+alow)/L (h+ahig)/L])

for j=1:N2+1
  plot(x(1:N1+1,j)/L,y(1:N1+1,j)/L,'k')
end

for i=1:N1+1
  plot(x(i,1:N2+1)/L,y(i,1:N2+1)/L,'k')
end

%axis([-0.5 0.5 -0.8 0.8])
%plot(0.2,0.2,'ko')

end

%---
% covariant base vectors
%---

for j=2:N2+1
 for i=2:N1+1
  gcov1_x(i,j) = (x(i+1,j)-x(i-1,j))/(2.0*Dx1);
  gcov1_y(i,j) = (y(i+1,j)-y(i-1,j))/(2.0*Dx1);
  gcov2_x(i,j) = (x(i,j+1)-x(i,j-1))/(2.0*Dx2);
  gcov2_y(i,j) = (y(i,j+1)-y(i,j-1))/(2.0*Dx2);
 end
end

for i=2:N1+1
  gcov1_x(i,1) = (x(i+1,1)-x(i-1,1))/(2.0*Dx1);
  gcov1_y(i,1) = (y(i+1,1)-y(i-1,1))/(2.0*Dx1);
  gcov2_x(i,1) = (x(i,2)-x(i,1))/Dx2;
  gcov2_y(i,1) = (y(i,2)-y(i,1))/Dx2;
end

%---
% wrap
%---

for j=1:N2+1
  gcov1_x(1,j) = gcov1_x(N1+1,j);
  gcov1_y(1,j) = gcov1_y(N1+1,j);
  gcov2_x(1,j) = gcov2_x(N1+1,j);
  gcov2_y(1,j) = gcov2_y(N1+1,j);
end

%---
% plot the covariant base vectors
%---

iplot31 = 0;
iplot31 = 1;

if(iplot31==1)

figure(31)
hold on
xlabel('x/L','fontsize',15)
ylabel('y/L','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-0.5 0.5 -1.1*(h+alow)/L 1.1*(h+ahig)/L])

 plot(x(1:N1+1,1)/L,y(1:N2+1,1)/L,'k')
%plot(x(1:N1+1,N2+1)/L,y(1:N2+1,N2+1)/L,'k')

fc = 0.75*Dx1;

for j=1:N2+1
 for i=1:N1+1
  arrx = fc*gcov1_x(i,j);
  arry = fc*gcov1_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  gaidaros = gaidaros/L;
  plot(gaidaros(:,1), gaidaros(:,2),'k')
  arrx = fc*gcov2_x(i,j);
  arry = fc*gcov2_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  gaidaros = gaidaros/L;
  plot(gaidaros(:,1), gaidaros(:,2),'k')
 end
end

end

%---
% metric coefficients
%---

for j=1:N2+1
 for i=1:N1+1
  covmet11(i,j) = gcov1_x(i,j)*gcov1_x(i,j) ...
                + gcov1_y(i,j)*gcov1_y(i,j);
  covmet12(i,j) = gcov1_x(i,j)*gcov2_x(i,j) ...
                + gcov1_y(i,j)*gcov2_y(i,j);
  covmet21(i,j) = covmet12(i,j);
  covmet22(i,j) = gcov2_x(i,j)*gcov2_x(i,j) ...
                + gcov2_y(i,j)*gcov2_y(i,j);
  covg(i,j) = covmet11(i,j)*covmet22(i,j)-covmet12(i,j)^2;
  srcovg(i,j) = sqrt(covg(i,j));

  covmet = [ covmet11(i,j), covmet12(i,j);...
             covmet21(i,j), covmet22(i,j)];

  invcovmet = inv(covmet);

  conmet11(i,j) = invcovmet(1,1);
  conmet12(i,j) = invcovmet(1,2);
  conmet21(i,j) = invcovmet(2,1);
  conmet22(i,j) = invcovmet(2,2);
  cong(i,j) = conmet11(i,j)*conmet22(i,j)-conmet12(i,j)^2;

 end
end

%---
% wrap
%---

for j=1:N2+1
 covmet11(N1+2,j) = covmet11(2,j);
 covmet12(N1+2,j) = covmet12(2,j);
 covmet21(N1+2,j) = covmet21(2,j);
 covmet22(N1+2,j) = covmet22(2,j);
     covg(N1+2,j) =     covg(2,j);
   srcovg(N1+2,j) =   srcovg(2,j);
 conmet11(N1+2,j) = conmet11(2,j);
 conmet12(N1+2,j) = conmet12(2,j);
 conmet21(N1+2,j) = conmet21(2,j);
 conmet22(N1+2,j) = conmet22(2,j);
     cong(N1+2,j) =     cong(2,j);
end

%---
% contravariant base vectors
%---

for j=1:N2+1
 for i=1:N1+1
  gcon1_x(i,j) = ( covmet22(i,j)*gcov1_x(i,j) ...
                  -covmet12(i,j)*gcov2_x(i,j))/covg(i,j);
  gcon1_y(i,j) = ( covmet22(i,j)*gcov1_y(i,j) ...
                  -covmet12(i,j)*gcov2_y(i,j))/covg(i,j);
  gcon2_x(i,j) = (-covmet12(i,j)*gcov1_x(i,j) ...
                  +covmet11(i,j)*gcov2_x(i,j))/covg(i,j);
  gcon2_y(i,j) = (-covmet12(i,j)*gcov1_y(i,j) ...
                  +covmet11(i,j)*gcov2_y(i,j))/covg(i,j);
 end
end

%---
% wrap
%---

for j=1:N2+1
 gcon1_x(N1+2,j) = gcon1_x(2,j);
 gcon1_y(N1+2,j) = gcon1_y(2,j);
 gcon2_x(N1+2,j) = gcon2_x(2,j);
 gcon2_y(N1+2,j) = gcon2_y(2,j);
end

%---
% plot the contravariant base vectors
%---

iplot32 = 0;
iplot32 = 1;

if(iplot32==1)

figure(32)
hold on
xlabel('x/L','fontsize',15)
ylabel('y/L','fontsize',15)
set(gca,'fontsize',15)
box on
axis equal
axis([-0.5 0.5 -1.1*(h+alow)/L 1.1*(h+ahig)/L])
%plot(x(1:N1+1,1)/L,y(1:N2+1,1)/L,'k')
%plot(x(1:N1+1,N2+1)/L,y(1:N2+1,N2+1)/L,'k')

fc = 0.75*Dx1;

for j=1:N2+1
 for i=1:N1+1
  arrx = fc*gcon1_x(i,j);
  arry = fc*gcon1_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  gaidaros = gaidaros/L;
  plot(gaidaros(:,1), gaidaros(:,2),'k')
  arrx = fc*gcon2_x(i,j);
  arry = fc*gcon2_y(i,j);
  gaidaros = arrow_cp(x(i,j),y(i,j),arrx,arry);
  gaidaros = gaidaros/L;
  plot(gaidaros(:,1), gaidaros(:,2),'k')
 end
end

end

%----
% optional
%----

Ido = 1;
Ido = 0;

if(Ido==1)

for j=1:N2+1
 for i=1:N1+1
  conmet11(i,j) = gcon1_x(i,j)*gcon1_x(i,j) ...
                + gcon1_y(i,j)*gcon1_y(i,j);
  conmet12(i,j) = gcon1_x(i,j)*gcon2_x(i,j) ...
                + gcon1_y(i,j)*gcon2_y(i,j);
  conmet21(i,j) = conmet12(i,j);
  conmet22(i,j) = gcon2_x(i,j)*gcon2_x(i,j) ...
                + gcon2_y(i,j)*gcon2_y(i,j);
  cong(i,j) = conmet11(i,j)*conmet22(i,j)-conmet12(i,j)^2;
 end
end

%---
% wrap
%---

for j=1:N2+1
 conmet11(N1+2,j) = conmet11(2,j);
 conmet12(N1+2,j) = conmet12(2,j);
 conmet21(N1+2,j) = conmet21(2,j);
 conmet22(N1+2,j) = conmet22(2,j);
     cong(N1+2,j) =     cong(2,j);
end

end

%===
% plot g and covg
%===

iplot35=1;
iplot35=0;

if(iplot35==1)

figure(35)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
view(-12,30)

%surf(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),g(1:N1+1,1:N2+1))
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),covg(1:N1+1,1:N2+1))

end

%---
% plot 1/g
%---

iplot36=1;
iplot36=0;

if(iplot36==1)

figure(36)
hold on
xlabel('x^1','fontsize',15)
ylabel('x^2','fontsize',15)
zlabel('1/g','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
view(-12,30)

mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),cong(1:N1+1,1:N2+1))

end

%---
% plot gcon_11, gcon_12, gcon_22
%---

iplot11=1;
iplot11=0;

if(iplot11==1)

figure(11)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^{\xi\xi}','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet11(1:N1+1,1:N2+1))
view(-12,30)

end

iplot12=1;
iplot12=0;

if(iplot12==1)

figure(12)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^{\xi\eta}','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet12(1:N1+1,1:N2+1))
view(-12,30)

end

iplot22=0;
iplot22=1;

if(iplot22==1)

figure(22)
hold on
xlabel('\xi','fontsize',15)
ylabel('\eta','fontsize',15)
zlabel('g^{\eta\eta}','fontsize',15)
set(gca,'fontsize',15)
box on
%axis equal
mesh(x1(1:N1+1,1:N2+1),x2(1:N1+1,1:N2+1),conmet22(1:N1+1,1:N2+1))
view(-12,30)

end

%-------------
% compute the effective velocities v1 and v2
%-------------

for j=2:N2
 for i=2:N1+1
  v1(i,j) = (conmet11(i+1,j)*srcovg(i+1,j) ...
            -conmet11(i-1,j)*srcovg(i-1,j))/(2.0*Dx1) ...
           +(conmet12(i,j+1)*srcovg(i,j+1) ...
            -conmet12(i,j-1)*srcovg(i,j-1))/(2.0*Dx2);
  v2(i,j) = (conmet21(i+1,j)*srcovg(i+1,j) ...
            -conmet21(i-1,j)*srcovg(i-1,j))/(2.0*Dx1) ...
           +(conmet22(i,j+1)*srcovg(i,j+1) ...
            -conmet22(i,j-1)*srcovg(i,j-1))/(2.0*Dx2);
  v1(i,j) = v1(i,j)/srcovg(i,j);
  v2(i,j) = v2(i,j)/srcovg(i,j);
 end
end

%---
% wrap
%---

for j=2:N2
  v1(1,j) = v1(N1+1,j);
  v2(1,j) = v2(N1+1,j);
end

for i=1:N1+1
  v1(i,1) = v1(i,2);
  v2(i,1) = v2(i,2);
  v1(i,N2+1) = v1(i,N2);
  v2(i,N2+1) = v2(i,N2);
end

%---------
% solve the Poisson equation
%---------

for j=1:N2+1
 for i=1:N1+2
  source(i,j) = 0.0;
  if(ichannel==1)
    source(i,j) = 5.0; % typical
  elseif(ichannel==2)
    source(i,j) = 1.0; % typical
  end
 end
end

for i=1:N1+2
  z(i) = 0.0; % typical
  v(i) = 0.1; % typical
  z(i) = 0.5; % typical
  v(i) = 0.0; % typical
end

%---
% generate the linear system
%---

[mats,mat,rhs] = pois_fds_PPDD  ...
  ...
   (a1,b1 ...
   ,a2,b2 ...
   ,N1,N2 ...
   ,conmet11,conmet12,conmet22 ...
   ,v1,v2 ...
   ,source ...
   ,z,v ...
   );

%-----------------
% printing session
%-----------------

%  mat

%---
% solution
%---

  sol = rhs/mat';

%---
% distribute the solution
%---

 p = 0;     % counter

 for j=2:N2
   for i=2:N1+1
     p = p+1;
     f(i,j) = sol(p);
   end
 end

  for i=1:N1+2  % boundary conditions
    f(i,   1) = z(i);
    f(i,N2+1) = v(i);
  end

  for j=1:N2+1   % wrap
    f(1,j) = f(N1+1,j);
  end

%---
% plot
%---

figure(91)

mesh(x(1:N1+1,1:N2+1),y(1:N1+1,1:N2+1),f(1:N1+1,1:N2+1))
%axis([0 2 0 1 0 0.1])
axis equal
box on
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
zlabel('f','fontsize',15)
set(gca,'fontsize',15)
view(-32,32)
