clear all
close all

%=================================
% Solution of the Poisson equation
%
% laplacian(f) + source = 0
%
% in a rectangle confined between
%
% -a < x < a and -b < y < b
%
% with the Neumann
% boundary condition all around
%
% -df/dx = w at the left
%  df/dx = q at the right
% -df/dy = z at the bottom
%  df/dy = v at the top
%=================================

Nx = 24;  % divisions
Ny = 16;

Nx = 2;
Ny = 2;

Nx = 16;
Ny = 14;

a = 1.0;
b = 1.0;

a = 1.0;
b = 0.5;

%---
% prepare
%---

Dx = 2.0*a/Nx;
Dy = 2.0*b/Ny;

beta = Dx^2/Dy^2;

Dx2 = 2.0*Dx;
Dy2 = 2.0*Dy;

%---
% grid
%---

for i=1:Nx+1
 x(i) = -a + (i-1)*Dx;
end

for i=1:Ny+1
 y(i) = -b + (i-1)*Dy;
end

%---
% Neumann boundary conditions
%---

for i=1:Nx+1
 z(i) =  0.1; % bottom
 v(i) = -0.1; % top
end

for j=1:Ny+1
 w(j) =  0.2; % left
 q(j) = -0.2; % right
end

%---
% specify the source (typical)
%---

for j=1:Ny+1
 for i=1:Nx+1
  source(i,j) = 20.0*x(i)*y(j);
 end
end

%---
% tridiagonal in x
%---

U(1,1) =-2;
U(1,2) = 2; 
U(Nx+1,Nx+1) =-2;
U(Nx+1,Nx)   = 2;

for i=2:Nx
 U(i,i)   =-2;
 U(i,i-1) = 1;
 U(i,i+1) = 1;
end

%---
% tridiagonal in y
%---

V(1,1) =-2;
V(1,2) = 2;
V(Ny+1,Ny+1) =-2;
V(Ny+1,Ny)   = 2;

for i=2:Ny
 V(i,i) = -2;
 V(i,i-1) = 1;
 V(i,i+1) = 1;
end

%---
% coefficient matrix
%---

A = kron(eye(Ny+1),U) + beta*kron(V,eye(Nx+1));
%A

%---
% system size
%---

Nsys = (Nx+1)*(Ny+1);

%---
% compose the adjoint eigenvector
% of the null eigenvalue
% as a Kronecker tensor product
%---

eigvx(1) = 0.5;

for i=2:Nx
 eigvx(i) = 1.0;
end

eigvx(Nx+1) = 0.5;

eigvy(1) = 0.5;

for j=2:Ny
 eigvy(j) = 1.0;
end
eigvy(Ny+1) = 0.5;

eigv = kron(eigvy',eigvx');

%v
%A'*v

%---
% normalize the eigenevector
%---

vnorm = 0;

for i=1:Nsys
 vnorm = vnorm + eigv(i)^2;
end

vnorm = sqrt(vnorm);

for i=1:Nsys
 eigvn(i) = eigv(i)/vnorm;
end

%---
% define the normalized eigenvector
% of the null eigenvalue
%---

for i=1:Nsys
 eigun(i) = 1.0/sqrt(Nsys);
end

%---
% right-hand side
%---

Ic = 0; % counter

for j=1:Ny+1
 for i=1:Nx+1

  Ic = Ic +1;

  rhs(Ic) = -source(i,j)*Dx^2;

  if(j==1)
    rhs(Ic) = rhs(Ic) - beta*Dy2*z(i);
  end

  if(j==Ny+1)
    rhs(Ic) = rhs(Ic) - beta*Dy2*v(i);
  end

  if(i==1)
    rhs(Ic) = rhs(Ic) - Dx2*w(j);
  end

  if(i==Nx+1)
    rhs(Ic) = rhs(Ic) - Dx2*q(j);
  end

 end
end

%---
% regularize the right-hand side
%---

rhsdotvn = 0.0;

for i=1:Nsys
 rhsdotvn = rhsdotvn + rhs(i)*eigvn(i);
end

for i=1:Nsys
 rhsn(i) = rhs(i) - rhsdotvn*eigvn(i);
end

%---
% solve
%---

%sol = rhs/A';

Areduced   = A(1:Nsys-1,1:Nsys-1);
rhsreduced = rhsn(1:Nsys-1);
sol    = rhsreduced/Areduced';
sol(Nsys) = 0.0;

residual = A*sol'-rhsn';
%residual

%---
% solution by the Moore--Penrose inverse
%---

MPinv = pinv(A);

MPsol = MPinv*rhs';

%MPsol = MPinv*rhsn';

shift = MPsol(Nsys);

for i=1:Nsys
 MPsol(i) = MPsol(i)-shift;
end

%---
% Moore--Penrose inverse properties
% (optional)
%---

%MPinv*eigvn'
%MPinv'*eigun'

for i=1:Nsys
 for j=1:Nsys
  prj(i,j) = -eigvn(i)*eigvn(j);
  prg(i,j) = -1.0/Nsys;
  prq(i,j) = eigvn(i)*eigun(j);
  prr(i,j) = eigun(i)*eigvn(j);
 end
 prj(i,i) = 1.0+prj(i,i);
 prg(i,i) = 1.0+prg(i,i);
end

%A*MPinv-prj
%MPinv*A-prg

MPinv1 = inv(A+prq)-prr;
MPdiff = MPinv1-MPinv;
MPdiff;
%max(MPdiff)

%---
% distribute the solution
%---

Ic = 0;

for j=1:Ny+1
 for i=1:Nx+1
  Ic = Ic+1;
  f(i,j) = sol(Ic);
  fMP(i,j) = MPsol(Ic);
 end
end

%---
% plot
%---

figure(1)
hold on
surf(x,y,f')
%surf(x,y,fMP')
%mesh(x,y,f')
%mesh(x,y,fMP')
box on
axis equal
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
zlabel('f','fontsize',15)
set(gca,'fontsize',15)
view(45,30)
