clear all
close all

%==================================
% Solution of Poisson equation
%
% laplacian(f) + source = 0
%
% in a rectangle confined between
% -a < x < a and -b < y < b
% with DNDD boundary conditions
% Solution of the Poisson equation:
%
% where D stands for Dirichlet
%   and N stands for Neumann
%
% f     = w at the left
% df/dx = q at the right
% f     = z at the bottom
% f     = v at the top
%
% beta = Dx^2/Dy^2
%==================================

Nx = 2*16;  % divisions
Ny = 2*12;

a = 1.0;
b = 0.8;

%---
% prepare
%---

Dx = 2.0*a/Nx;
Dy = 2.0*b/Ny;

beta = Dx^2/Dy^2;

Dx2 = 2.0*Dx;

%---
% grid
%---

for i=1:Nx+1
 x(i) = -a + (i-1)*Dx;
end

for i=1:Ny+1
 y(i) = -b + (i-1)*Dy;
end

%---
% boundary conditions (typical)
%---

for i=1:Nx+1
 z(i) = 0.2; % bottom (Dirichlet)
 v(i) = 0.1; % top    (Dirichlet)
end

for j=1:Ny+1
 w(j) = 0.0; % left  (Dirichlet)
 q(j) = 2.0; % right (Neumann)
end

%---
% specify the source (typical)
%---

for j=2:Ny
 for i=2:Nx+1
  source(i,j) = 10.0*x(i)*y(j);
 end
end

%---
% tridiagonal in x
%---

U(1,1) = -2;
U(1,2) = 1;
U(Nx,Nx) = -2;
U(Nx,Nx-1) = 2;  % notice the 2

for i=2:Nx-1
 U(i,i) = -2;
 U(i,i-1) = 1;
 U(i,i+1) = 1;
end

%---
% tridiagonal in y
%---

V(1,1) = -2;
V(1,2) = 1;
V(Ny-1,Ny-1) = -2;
V(Ny-1,Ny-2) = 1;

for i=2:Ny-2
 V(i,i) = -2;
 V(i,i-1) = 1;
 V(i,i+1) = 1;
end

%---
% coefficient matrix
%---

A = kron(eye(Ny-1),U) + beta*kron(V,eye(Nx));
%A

%---
% right-hand side
%---

Ic = 0; % counter

for j=2:Ny
 for i=2:Nx+1

  Ic = Ic+1;

  rhs(Ic) = source(i,j)*Dx^2;

  if(j==2)
    rhs(Ic) = rhs(Ic) - beta*z(i);
  end

  if(j==Ny)
    rhs(Ic) = rhs(Ic) - beta*v(i);
  end

  if(i==2)
    rhs(Ic) = rhs(Ic) - w(j);
  end

  if(i==Nx+1)
    rhs(Ic) = rhs(Ic) - Dx2*q(j);
  end

 end
end

%---
% solve
%---

SOL = rhs/A';

%---
% distribute
%---

Ic = 0;

for j=2:Ny
 for i=2:Nx+1
  Ic = Ic+1;
  f(i,j) = SOL(Ic);
 end
end

%---
% boundary conditions
%---

for i=1:Nx+1
 f(i,1) = z(i);    % bottom
 f(i,Ny+1) = v(i); % top
end

for j=1:Ny+1
 f(1,j) = w(j);     %  left 
end

%---
% plot
%---

figure(1)
mesh(x,y,f')
surf(x,y,f')
box on
axis equal
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
zlabel('f','fontsize',15)
set(gca,'fontsize',15)
view(-46,31)
