clear all
close all

%============================
% Solution of the Poisson equation:
%
% laplacian(f) + source = 0
%
% in a rectangle confined between
%
% -a < x < a and -b < y < b
%
% with DDDD boundary conditions
% where D stands for Dirichlet
%
% f = w at the left
% f = q at the right
% f = z at the bottom
% f = v at the top
%============================

Nx = 16;  % divisions
Ny = 14;

a = 0.50;  % typical
b = 0.40;  % typical

%---
% prepare
%---

Dx = 2.0*a/Nx;
Dy = 2.0*b/Ny;

beta = Dx^2/Dy^2;

%---
% grid
%---

for i=1:Nx+1
 x(i) = -a + (i-1)*Dx;
end

for j=1:Ny+1
 y(j) = -b + (j-1)*Dy;
end

%---
% boundary conditions (typical)
%---

for i=1:Nx+1
 z(i) = 1.0-1.2*i*(Nx+1-i)/Nx^2; % bottom
 v(i) = 1.0; % top
end

for j=1:Ny+1
 w(j) = 1.0; % left
 q(j) = 1.0; % right
end

%---
% specify the source (typical)
%---

for j=2:Ny
 for i=2:Nx
  source(i,j) = 200.0*x(i)*y(j);
 end
end

%---
% tridiagonal in x
%---

U(1,1) =-2;
U(1,2) = 1;
U(Nx-1,Nx-1) =-2;
U(Nx-1,Nx-2) = 1;

for i=2:Nx-2
 U(i,i) = -2;
 U(i,i-1) = 1;
 U(i,i+1) = 1;
end

%---
% tridiagonal in y
%---

V(1,1) =-2;
V(1,2) = 1;
V(Ny-1,Ny-1) =-2;
V(Ny-1,Ny-2) = 1;

for i=2:Ny-2
 V(i,i) = -2;
 V(i,i-1) = 1;
 V(i,i+1) = 1;
end

%---
% coefficient matrix: (Nx-1)*(Ny-1)
%---

A = kron(eye(Ny-1),U) + beta*kron(V,eye(Nx-1));
%A

%---
% right-hand side
%---

Ic = 0; % counter

for j=2:Ny
 for i=2:Nx

  Ic = Ic +1;

  rhs(Ic) = -source(i,j)*Dx^2;

  if(j==2)
    rhs(Ic) = rhs(Ic) - beta*z(i);
  end

  if(j==Ny)
    rhs(Ic) = rhs(Ic) - beta*v(i);
  end

  if(i==2)
    rhs(Ic) = rhs(Ic) - w(j);
  end

  if(i==Nx)
    rhs(Ic) = rhs(Ic) - q(j);
  end

 end
end

%---
% solve
%---

SOL = rhs/A';

%---
% distribute the solution
%---

Ic = 0;

for j=2:Ny
 for i=2:Nx
  Ic = Ic+1;
  f(i,j) = SOL(Ic);
 end
end

%---
% boundary conditions
%---

for i=1:Nx+1
 f(i,1) = z(i);    % bottom
 f(i,Ny+1) = v(i); % top
end

for j=1:Ny+1
 f(1,j) = w(j);     % left 
 f(Nx+1,j) = q(j);  % right
end

%---
% plot
%---

figure(1)
%mesh(x,y,f')
surf(x,y,f')
box on
axis equal
axis([-a a -b b -1.2*a a])
xlabel('x','fontsize',15)
ylabel('y','fontsize',15)
zlabel('f','fontsize',15)
set(gca,'fontsize',15)
view(-32,15)
