clear all
close all

%=====================================
% temperature evolution across two slabs
% by the implicit btcs method
%
% T1w : lower wall temperature
% T2w : upper wall temperature
%
% T1_phan = T1(N1+2)    : phantom node
% T2_phan = T2(0)       : phantom node
%=====================================

%---
% parameters
%---

h1 = 1.0;
h2 = 2.0*h1;

N1 = 16;
N2 = 48;

T1w = 0.0;
T2w = 1.0;

kappa1 = 1.0;
k1 = 1.0;

kappa2 = 3.0*kappa1;
k2 = 0.4*k1;
kc = 1.*k1/h1;

Dt = 0.05;   % time step

%---
% prepare
%---

Dy1 = h1/N1;
Dy2 = h2/N2;

alpha1 = kappa1*Dt/Dy1^2;
alpha2 = kappa2*Dt/Dy2^2;

beta = Dy2/Dy1;
lambda = k2/k1;
lambdac = kc*h1/k1;

h = h1+h2;

%---
% for steady state:
%---

xi = (T2w-T1w)/h/(k2/(h*kc)+(lambda-1)*h1/h+1.0);

%---
% grid, initial conditions, steady state
%---

for i=1:N1+1
 y1(i) = (i-1)*Dy1;
 T1(i) = 0;
 T1steady(i) = T1w + lambda*xi*y1(i);
end

T1(1) = T1w;

for i=1:N2+1
 y2(i) = h1+(i-1)*Dy2;
 T2(i) = 0;
 T2steady(i) = T2w + xi*(y2(i)-h);
end

T2(N2+1) = T2w;

%---
% graphics
%---

figure(1)
hold on
box on
axis([0 1 0 h1+h2])
set(gca,'fontsize',15)
xlabel('T','fontsize',15)
ylabel('y','fontsize',15)
plot([0 2],[h1 h1],'k')
plot(T1steady,y1,'r-');
plot(T2steady,y2,'k-');

%---
% time stepping
%---

t = 0.0;

nstep = 16;

%---
for step=1:nstep
%---

plot(T1,y1,'.r-');
plot(T2,y2,'.k-');

if(step==1)
%  figure(1)
%  Handle1 = plot(T1,y1,'ro-');
%  Handle2 = plot(T2,y2,'ko-');
%  set(Handle1, 'erasemode', 'xor');
%  set(Handle2, 'erasemode', 'xor');
else
% set(Handle1,'XData',T1,'YData',y1);
% set(Handle2,'XData',T2,'YData',y2);
% pause(0.1)
% drawnow
end


T1save = T1;
T2save = T2;

T1_phan = 0.0;
T2_phan = 0.0;

[T1,T2,Phi1,Phi2] = slabs_solve ...
  ...
         (T1,alpha1,T1w,T1_phan,N1 ...
         ,T2,alpha2,T2w,T2_phan,N2 ...
         ,beta,lambda,lambdac);

RHS(1) = -Phi1;
RHS(2) = -Phi2;

T1_phan = 1.0;
T2_phan = 0.0;

T1 = T1save;
T2 = T2save;

[T1,T2,Phi1,Phi2] = slabs_solve ...
  ...
    (T1,alpha1,T1w,T1_phan,N1 ...
    ,T2,alpha2,T2w,T2_phan,N2 ...
    ,beta,lambda,lambdac);

AMAT(1,1) = Phi1+RHS(1);
AMAT(2,1) = Phi2+RHS(2);

T1_phan = 0.0;
T2_phan = 1.0;

T1 = T1save;
T2 = T2save;

[T1,T2,Phi1,Phi2] = slabs_slabs ...
 ...
        (T1,alpha1,T1w,T1_phan,N1 ...
        ,T2,alpha2,T2w,T2_phan,N2 ...
        ,beta,lambda,lambdac);

AMAT(1,2) = Phi1+RHS(1);
AMAT(2,2) = Phi2+RHS(2);

PHAN = RHS/AMAT';

T1_phan = PHAN(1);
T2_phan = PHAN(2);

T1 = T1save;
T2 = T2save;

[T1,T2,Phi1,Phi2] = slabs_solve ...
 ...
     (T1,alpha1,T1w,T1_phan,N1 ...
     ,T2,alpha2,T2w,T2_phan,N2 ...
     ,beta,lambda,lambdac);

t = t+Dt;

[t, Phi1, Phi2]

%---
end
%---
