clear all
close all

%==========================
% Equilibrium shell shapes
% in two dimensions
%==========================

%---
% parameters
%---

a = 1;
kappaB = 1.0;
Nstp = 2*64;   % number of integration steps
Niter = 50;  % number of Newton iterations
mode = 2;
eps = 0.001;  % for the Jacobian
tol = 0.000001; % iteration tolerance

%---
% prepare
%---

Dl = 2*pi*a/(mode*Nstp);

%---
% graphics
%---

figure(1)
hold on
xlabel('x/a','fontsize',15)
ylabel('y/a','fontsize',15)
set(gca,'fontsize',15)
axis equal
box on
axis([-1.5 1.5 -0.5 2.5])

%---
% loop over transmural pressures
%---

%---
for iloop=1:8
%---

%---
% initial guesses for c and kappa0
%---

if(iloop==1)
 Dp = 3.0;
 fcc =  1.01;
 fck = 0.999;
end

if(iloop==2)
 Dp = 3.1;
 fcc = 0.9289;
 fck = 0.5315;
end

if(iloop==3)
 Dp = 3.2;
 fcc =  0.8195;
 fck = 0.1879;
end

if(iloop==4)
 Dp  = 3.5;
 fcc =  0.6087;
 fck = -0.1131;
end

if(iloop==5)
 Dp = 4.0;
 fcc = 0.4351;
 fck = -0.4943;
end

if(iloop==6)
 Dp = 4.5;
 fcc = 0.4351;
 fck = -0.4943;
end

if(iloop==7)
 Dp = 5.0;
 fcc = 0.4351;
 fck = -0.4943;
end

if(iloop==8)
 Dp = 5.247;
 fcc = 0.4351;
 fck = -0.4943;
end

c_circle = 2*Dp*a^3/kappaB - 1.0;
kappa0_circle = 1.0/a;

c = fcc*c_circle;
kappa0 = fck*kappa0_circle;

%----------------------------
% start the Newton iterations
%----------------------------

   for Iter=1:Niter

    sarray = shell_ode(a,c,kappa0,Dp ...
                          ,kappaB,Nstp,Dl);

    f(1) = sarray(7,Nstp+1)-2*pi/mode;
    f(2) = sarray(1,Nstp+1)-kappa0;

%---------------------
% compute the Jacobian
% by numerical differentiation
%---------------------

    c = c+eps;      % perturb
    sarray = shell_ode(a,c,kappa0,Dp ...
                      ,kappaB,Nstp,Dl);
    c = c-eps;      % reset

    f1(1) = sarray(7,Nstp+1)-2*pi/mode;
    f1(2) = sarray(1,Nstp+1)-kappa0;
    Jac11 = (f1(1)-f(1))/eps;
    Jac21 = (f1(2)-f(2))/eps;

    kappa0 = kappa0+eps;      % perturb
    sarray = shell_ode(a,c,kappa0,Dp ...
                      ,kappaB,Nstp,Dl);
    kappa0 = kappa0-eps;      % reset
    f1(1) = sarray(7,Nstp+1)-2*pi/mode;
    f1(2) = sarray(1,Nstp+1)-kappa0;
    Jac12 = (f1(1)-f(1))/eps;
    Jac22 = (f1(2)-f(2))/eps;

%---
% solve the equation: Jac . Dx = - f
% for the correction vector Dx
% by Cramer's rule
%---

     b1  = -f(1);
     b2  = -f(2);
     Det = Jac11*Jac22-Jac12*Jac21;
     dc = (b1*Jac22-Jac12*b2)/Det;
     dkappa0 = (b2*Jac11-Jac21*b1)/Det;

%--------
% correct
%--------

     c = c + dc;
     kappa0 = kappa0 + dkappa0;
     c;
     Iter
     [c kappa0]

%-------
% escape
%-------

     iescape = 1;
     if(abs(dc) > tol) iescape = 0; end
     if(abs(dkappa0) > tol) iescape = 0; end

     if(iescape==1)
       Iflag = 0;
       sarray = shell_ode(a,c,kappa0,Dp ...
                         ,kappaB,Nstp,Dl);
       f(1) = sarray(7,Nstp+1)-2*pi/mode;
       f(2) = sarray(1,Nstp+1)-kappa0;
       break
     end

%----
 end  % of iterations
%----

% shape aspect ratio

 xs = 0.0;
 ys = 0.0;

 xshift = 0.0;
 yshift = 0.5*(sarray(4,Nstp+1)-sarray(4,1));

 for i=1:Nstp
  xs = xs + (sarray(3,i)-xshift)^2;
  ys = ys + (sarray(4,i)-yshift)^2;
 end
 aspect(iloop) = sqrt(ys/xs);
 Dpplot(iloop) = Dp*a^3/kappaB;

 plot( sarray(3,:),sarray(4,:),'k')
 plot(-sarray(3,:),sarray(4,:),'k')

%---
end % of iloop over Dp
%---

figure(2)
hold on
plot(Dpplot,aspect,'ko-')
xlabel('\Deltap a^3/\kappa_B','fontsize',15)
ylabel('aspect ratio','fontsize',15)
set(gca,'fontsize',15)
axis([3 6 0.2 1])
box on

for i=1:32
 dplot(i) = 3+(i-1)*3.0/32;
 aob(i) = exp(-2.0*(dplot(i)-3));
 aob(i) = 1.0-0.5*sqrt(dplot(i)-3);
end
plot(dplot,aob,'k--')

