clear all
close all

%=====================================
% velocity profile evolution
% across two layers
% in unidirectional flow
% computed by the implicit btcs method
%
% u1w : lower wall temperature
% u2w : upper wall temperature
%
% u1_phan = u1(N1+2)    : phantom node
% u2_phan = u2(0)       : phantom node
%=====================================

%---
% parameters
%---

h1 = 1.0;
h2 = 2.0*h1;

N1 = 16;
N2 = 48;

u1w = 0.0;
u2w = 0.8;

mu1 = 1.0;
rho1 = 1.0;

mu2 = 0.7*mu1;
rho2 = 0.4*rho1;

chi =-0.4;
gx = 0.0;

Dt = 0.5;   % time step

%---
% prepare
%---

Dy1 = h1/N1;
Dy2 = h2/N2;

nu1 = mu1/rho1;
nu2 = mu2/rho1;

alpha1 = nu1*Dt/Dy1^2;
alpha2 = nu2*Dt/Dy2^2;

h = h1+h2;
beta = Dy2/Dy1;
lambda = mu2/mu1;
vr = rho2/rho1;
hr = h2/h1;

gamma = lambda/vr;
tmp = gamma+beta*lambda;
a1 = (gamma-beta*lambda)/tmp;
a2 = 2.0*lambda*(beta*beta-gamma)/(beta*tmp);
a3 = 2.0*gamma*lambda/(beta*tmp);
a4 = lambda*Dy2*Dy2*(1.0-1.0/vr)/(beta*mu1*tmp);

%---
% grid, initial conditions
%---

for i=1:N1+1
 y1(i) = (i-1)*Dy1;
 u1(i) = 0;
end

u1(1) = u1w;

for i=1:N2+1
 y2(i) = h1+(i-1)*Dy2;
 u2(i) = 0;
end

u2(N2+1) = u2w;

%---
% graphics
%---

figure(1)
hold on
box on
axis([-1 1 0 h1+h2])
set(gca,'fontsize',15)
xlabel('u','fontsize',15)
ylabel('y','fontsize',15)
plot([-1 1],[h1 h1],'k')

%---
% time stepping
%---

t = 0.0;

nstep = 128;

%---
for step=1:nstep
%---

plot(u1,y1,'.r-');
plot(u2,y2,'.k-');

if(step==1)
%  figure(1)
%  Handle1 = plot(T1,y1,'ro-');
%  Handle2 = plot(T2,y2,'ko-');
%  set(Handle1, 'erasemode', 'xor');
%  set(Handle2, 'erasemode', 'xor');
else
% set(Handle1,'XData',T1,'YData',y1);
% set(Handle2,'XData',T2,'YData',y2);
% pause(0.1)
% drawnow
end

u1save = u1;
u2save = u2;

u1_phan = 0.0;
u2_phan = 0.0;

[u1,u2,Phi1,Phi2] = solve_layers ...
 ...
        (u1,alpha1,u1w,u1_phan,N1 ...
        ,u2,alpha2,u2w,u2_phan,N2 ...
        ,Dt,chi,rho1,rho2,gx ...
        ,beta,lambda ...
        ,a1,a2,a3,a4);

RHS(1) = -Phi1;
RHS(2) = -Phi2;

u1_phan = 1.0;
u2_phan = 0.0;

u1 = u1save;
u2 = u2save;

[u1,u2,Phi1,Phi2] = solve_layers ...
 ...
        (u1,alpha1,u1w,u1_phan,N1 ...
        ,u2,alpha2,u2w,u2_phan,N2 ...
        ,Dt,chi,rho1,rho2,gx ...
        ,beta,lambda ...
        ,a1,a2,a3,a4);

AMAT(1,1) = Phi1+RHS(1);
AMAT(2,1) = Phi2+RHS(2);

u1_phan = 0.0;
u2_phan = 1.0;

u1 = u1save;
u2 = u2save;

[u1,u2,Phi1,Phi2] = solve_layers ...
 ...
        (u1,alpha1,u1w,u1_phan,N1 ...
        ,u2,alpha2,u2w,u2_phan,N2 ...
        ,Dt,chi,rho1,rho2,gx ...
        ,beta,lambda ...
        ,a1,a2,a3,a4);

AMAT(1,2) = Phi1+RHS(1);
AMAT(2,2) = Phi2+RHS(2);

PHAN = RHS/AMAT';
u1_phan = PHAN(1);
u2_phan = PHAN(2);

u1 = u1save;
u2 = u2save;

[u1,u2,Phi1,Phi2] = solve_layers ...
 ...
        (u1,alpha1,u1w,u1_phan,N1 ...
        ,u2,alpha2,u2w,u2_phan,N2 ...
        ,Dt,chi,rho1,rho2,gx ...
        ,beta,lambda ...
        ,a1,a2,a3,a4);

t = t+Dt;

[t, Phi1, Phi2]

%---
end
%---
