function [flow_rate,u_mean,u_max,u] = tube_rec ...
 ...
   (a ...
   ,b ...
   ,visc ...
   ,den ...
   ,gac ...
   ,chi ...
   ,beta ...
   ,maxsm ...
   ,y ...
   ,z ...
   )

%===========================================
% SEMLIB
%
% Copyright by C. Pozrikidis, 1999
% All rights reserved
%
% This program is to be used only under the
% stipulations of the licensing agreement
%===========================================

%================================================
% Steady flow through a tube with rectangular
% cross-section of size (2a)X(2b)
% due to a pressure gradient or gravity
%
% The origin is set at the tube center
%
% SYMBOLS:
% --------
%
% beta:  tube inclination angle
% gac:   magnitude of the acceleration of gravity
% maxsm: truncation limit for infinite sum
%================================================

%--------
% prepare
%--------

  as  = a^2;
  bs  = b^2;
  aob = a/b;

  pg_mod = chi + den*gac*sin(beta);  % modified pressure gradient

  area = 4.0*a*b;  % tube cross-sectional area

%--------------------------
% compute:
%
% (a) velocity at (y,z)
% (b) flow rate 
% (c) max velocity
%--------------------------

  zs = z^2;

  sumv = 0.0;     % velocity
  summ = 0.0;     % max velocity
  sumf = 0.0;     % flow rate

  sign = -1.0;

  for n=1:maxsm

    alpha = (n-0.5)*pi;

    arg1 = alpha*y/b;
    arg2 = alpha*aob;
    arg3 = alpha*z/b;
    deno  = cosh(arg2)*alpha^3;
    sumv = sumv + sign*cosh(arg1)*cos(arg3)/deno;
    summ = summ + sign/deno;
    sumf = sumf + tanh(arg2)/alpha^5;

    sign = -sign;

  end

%---
% velocity
%---

  u = 0.5*pg_mod*(bs-zs + 4.0*bs*sumv)/visc;

%---
% flow rate
%---

  F = 1.0-6.0*sumf/aob;

  flow_rate = 4.0/3.0 *pg_mod *a*b^3 *F/visc;

  u_mean  = flow_rate/area;

%---
% max velocity
%---

  u_max = 0.5*pg_mod/visc * (bs+4.0*bs*summ);

%-----
% done
%-----

  return
